!* +-----------------------------------------------------------------------
!* | Test new algorithm for soil water limits.  From Ritchie, J. T., A.
!* | Gerakis, and A. Suleiman.  1999.  Simple model to estimate field-
!* | measured soil water limits. Trans. ASAE 42:1609-1614.
!* +-----------------------------------------------------------------------
!* | Created by: aris gerakis - jun 98
!* | Modified by: aris gerakis - feb 99:  added Ksat
!* | Modified by: aris gerakis - 2 nov 2000:  added updated citation
!* | Modified by: brian baer - 2023-11-14: change Ksat eqution to match web
!* |                                       version 
!* +-----------------------------------------------------------------------

program swlimits

implicit none

real        :: bd, cf, clay, dul, Ksat, ll, oc, pesw, sand, sat
character   :: ans, infile*12

write (unit = 6, fmt = 5000)
5000 format ( &
1x,'Test new algorithm for soil water limits and Ksat. From: (1) Ritchie, J. T.,', /, &
1x,'A. Gerakis, and A. Suleiman.  1999.  Simple model to estimate field-measured', /, &
1x,'soil water limits. Trans. ASAE 42:1609-1614, and (2) Suleiman, A. A., and,', /, &
1x,'J. T. Ritchie.  In review. Estimating saturated hydraulic conductivity from', /, &
1x,'drained upper limit, water content and bulk density.  2 nov 2000.')

ans = 'y'
bd = 0.0
cf = 0.0
clay = 0.0
dul = 0.0
Ksat = 0.0
ll = 0.0
oc = 0.0
pesw = 0.0
sand = 0.0
sat = 0.0

write (unit = 6, fmt = 8000)
8000 format (/, 1x, 'Do you want to run interactive mode? (y/n)  ', $)
read (unit = 5, fmt = *) ans

if (ans .eq. 'y' .or. ans .eq. 'Y') then
   do while (ans .eq. 'y' .or. ans .eq. 'Y')
      write (unit = 6, fmt = 1000)
      1000 format (/, 1x, 'Enter sand%, clay%, BD(Mg/m3), CF%, OC% ', &
                   'separated by spaces: ')
      read (unit = 5, fmt = *) sand, clay, bd, cf, oc
      if (clay .eq. 0.0) then
         write (unit = 6, fmt = 2000)
         2000 format (1x, 'Clay cannot be 0, try again')
      elseif (sand .eq. 0.0) then
         write (unit = 6, fmt = 6000)
         6000 format (1x, 'Sand cannot be 0, try again')
      elseif (sand + clay .gt. 100.0) then
         write (unit = 6, fmt = 3000)
         3000 format (1x, 'Impossible combination of sand and clay, try again')
      elseif (bd .gt. 2.65) then
         write (unit = 6, fmt = 7000)
         7000 format (1x, 'Impossible BD, try again')
      else
         dul = bd * 0.186 * (sand/clay)**(-0.141)
         pesw = 0.132 - (2.5e-6) * exp(0.105*sand)
         ! Adjust for coarse fragments:
         if (cf .gt. 0.0) then
            dul = dul / (1.0 + (bd / 2.65) / (100.0 / cf - 1.0)) 
            pesw = pesw / (1.0 + (bd / 2.65) / (100.0 / cf - 1.0)) 
         endif
         ! Adjust for OC:
         dul = dul + 0.01 * oc
         pesw = pesw + 0.005 * oc
         ! Cannot exceed 92% of porosity:
         dul = amin1 (dul, 0.92 * (1.0 - bd/2.65))
         ! Cannot drop below 1%:
         if (dul - pesw .ge. 0.01) then
           ll = dul - pesw
         else
           ll = 0.01
           pesw = dul - ll
         endif
      
         ! Hydraulic conductivity:

         sat = 1.0 - bd / 2.65

         if (dul .gt. 0.0) then
            !  Modified by: aris gerakis - 20 nov 2000:  added Ayman's corrections
            !  75 changed to 37
            Ksat = (37.0 * ((sat - dul) / dul) ** 2)/24.0
         else
            write (unit = 6, fmt = 4100) dul
            4100 format (/, 1x, 'DUL cannot be ', f5.1)
         endif

         write (unit = 6, fmt = 4000) dul, ll, pesw, Ksat
         4000 format (1x, 'DUL = ', f4.2, 1x, 'LL = ', f4.2, 1x, 'PESW = ', &
                      f4.2, 1x, 'm3/m3', 1x, 'Ksat = ', f6.1, 1x, 'cm/d', &
                      /, 1x, 'Try another combination? (y/n)', $)
         read (unit = 5, fmt = *) ans
      endif
   enddo  ! end one soil
else ! if not interactive
   write (unit = 6, fmt = 9000)
   9000 format (/, &
   1x, 'Supply file with 5 columns, in free format: sand%, clay%, ', &
   'BD(Mg/m3), CF%, OC%.')
   ! write (unit = 6, fmt = '($a)') ' Enter input filename up to 12 characters: '
   write (6,*) ' Enter input filename up to 12 characters: '
   ! read (unit = 5, fmt = '(a)') infile
   read (5,*) infile
   open (unit = 10, file = infile, status = 'old')
   open (unit = 20, file = 'swlimits.out', status = 'unknown')
   write (unit = 20, fmt = *)'sand silt clay   bd   cf   oc   dul    ll  pesw   Ksat'
   ! do while (.not. eof(unit = 10))
   !    read (unit = 10, fmt = *) sand, clay, bd, cf, oc
   do while (.true.)
      read (unit = 10, fmt = *, end=100) sand, clay, bd, cf, oc
      if (clay .eq. 0.0) then
         write (unit = 6, fmt = 2010) sand, (100 - sand - clay), clay, bd, cf, oc
         write (unit = 20, fmt = 2010) sand, (100 - sand - clay), clay, bd, cf, oc
         2010 format (3(1x, f4.1), 1x, f4.2, 2(1x, f4.1), ' Clay cannot be 0')
      elseif (sand .eq. 0.0) then
         write (unit = 6, fmt = 6010) sand, (100 - sand - clay), clay, bd, cf, oc
         write (unit = 20, fmt = 6010) sand, (100 - sand - clay), clay, bd, cf, oc
         6010 format (3(1x, f4.1), 1x, f4.2, 2(1x, f4.1), ' Sand cannot be 0')
      elseif (sand + clay .gt. 100.0) then
         write (unit = 6, fmt = 3010) sand, (100 - sand - clay), clay, bd, cf, oc
         write (unit = 20, fmt = 3010) sand, (100 - sand - clay), clay, bd, cf, oc
         3010 format (3(1x, f4.1), 1x, f4.2, 2(1x, f4.1), &
                     ' Impossible combination of sand and clay')
      elseif (bd .gt. 2.65) then
         write (unit = 6, fmt = 7010) sand, (100 - sand - clay), clay, bd, cf, oc
         write (unit = 20, fmt = 7010) sand, (100 - sand - clay), clay, bd, cf, oc
         7010 format (3(1x, f4.1), 1x, f4.2, 2(1x, f4.1), ' Impossible BD')
      else
         dul = bd * 0.186 * (sand/clay)**(-0.141)
         pesw = 0.132 - (2.5e-6) * exp(0.105*sand)
         ! Adjust for coarse fragments:
         if (cf .gt. 0.0) then
            dul = dul / (1.0 + (bd / 2.65) / (100.0 / cf - 1.0)) 
            pesw = pesw / (1.0 + (bd / 2.65) / (100.0 / cf - 1.0)) 
         endif
         ! Adjust for OC:
         dul = dul + 0.01 * oc
         pesw = pesw + 0.005 * oc
         ! Cannot exceed 92% of porosity:
         dul = amin1 (dul, 0.92 * (1.0 - bd/2.65))
         ! Cannot drop below 1%:
         if (dul - pesw .ge. 0.01) then
           ll = dul - pesw
         else
           ll = 0.01
           pesw = dul - ll
         endif

         ! Hydraulic conductivity:

         sat = 1.0 - bd / 2.65

         if (dul .gt. 0.0) then
            !  Modified by: aris gerakis - 20 nov 2000:  added Ayman's corrections
            !  75 changed to 37
            Ksat = (37.0 * ((sat - dul) / dul) ** 2)/24.0
         else
            write (unit = 6, fmt = 2020) sand, (100 - sand - clay), clay, bd, cf, oc, dul
            write (unit = 20, fmt = 2020) sand, (100 - sand - clay), clay, bd, cf, oc, dul
            2020 format (3(1x, f4.1), 1x, f4.2, 2(1x, f4.1), ' DUL cannot be ', f5.1)
         endif

         write (unit = 20, fmt = 4010) sand, (100 - sand - clay), clay, bd, cf, oc, dul, ll, pesw, Ksat
         4010 format (3(1x, f4.1), 1x, f4.2, 2(1x, f4.1), 3(1x, f5.3), 1x, f6.1)
      endif
   enddo
   100 continue   
   write (unit = 6, fmt = '(a)') ' Your output file is swlimits.out'
   close (unit = 10)
   close (unit = 20)
endif ! end if non-interactive

end program swlimits
